# -*- coding: utf-8 -*-
"""
Action d'un filtre passe-bas d'ordre 1 sur un signal carré    
Filtre passe-bas d'ordre 1.
Facilement modifiable en passe-haut :
    multiplier le dénominateur de la fonction de transfert par jf/fc
"""

#### Importation des bibliothèques utiles

import numpy as np
import matplotlib.pyplot as plt

# Spectre du signal d'entrée rectangulaire (entre 0 et 2 V)
"""
On fabrique un signal carré à partir de sa décomposition spectrale sous forme d'une série de sinus
Ue = Uo + U1.sin(2pift) + U1/3.sin(2*pi*3f*t) + ...
"""

# Signal d'entrée = série de Fourier

fe = 1000                                    # fréquence du signal d'entrée (en Hz)     
n = 5		                                 # harmoniques impairs uniquement
B = [2*i + 1 for i in range(n)]              # les n premiers entiers impairs
F = np.array([0] + B)*fe                     # tableau de n+1 termes
U0 = 2                                       # valeur moyenne (V)
U1 = 1                                       # amplitude du fondamental (V)
Ue = np.array([U0] + list(U1/np.array(B)))   # amplitudes des composantes (V)
phie = np.array([0] + [-90]*n)	             # phases à l'origine (degrés)


# Constantes du filtre d'ordre 1
#---------------------------------------------------------------
fc = 100            # fréquence de coupure (Hz)
H0 = 1              # gain en basse fréquence
#---------------------------------------------------------------

# Fonction de transfert H = H0*num/(1 + j f/fc)
def H(H0, fc, f):
    #num = 1                     # passe-bas 
    num = 1j*f/fc              # passe-haut
    return H0 * num /(1 + 1j*f/fc)

# Action du filtre sur chaque raie du signal d'entrée
def ordre_1(H0, fc, F, U) :          # ici U est pour le moment une variable muette
    Hs   = H(H0, fc, F)
    Us   = np.abs(Hs) * U
    phis = np.angle(Hs, deg=True)
    return Us, phis

#Définition du diagramme de Bode
n = 2000                                # nombre de points
f = np.logspace(-2, 2, n+1)*fc          # n intervalles, entre 10^-2fc et 10^2fc ; n+1 valeurs
GdB = 20*np.log10(np.abs(H(H0, fc, f)))
phi = np.angle(H(H0, fc, f), deg=True)


#Tracé de GdB et phi en concordance de fréquence
plt.figure(1)
plt.subplot(211)
plt.semilogx(f, GdB, 'b')
plt.title('Diagramme de Bode, H0= ' + str(H0) + ', fc ='+ str(fc) + 'Hz')
plt.ylabel('GdB')
plt.grid()

plt.subplot(212)
plt.semilogx(f, phi, 'b')
plt.ylabel('phase (°)')
plt.xlabel('f (Hz)')
plt.grid()

#plt.show()

## Superposition des spectres d'entrée et sortie pour f0 choisie
                                        # on affecte Ue à U défini au dessus 
Us, phis = ordre_1(H0, fc, F, Ue)	    # listes des amplitudes de sortie et déphasages
largeur = fe/12                         # largeur des bâtons

plt.figure(2)
ax1 = plt.subplot(211)					                            # figure 2 lignes, 1 colonne
ax1.bar(F - largeur/2, Ue, largeur, label='Uem fe =' + str(fe) + 'Hz')                    # spectre de Ue
ax1.bar(F + largeur/2, Us, largeur, label= 'Usm pour fc = ' + str(fc) + 'Hz') # spectre de Us
ax1.set_ylabel('U (V)')
ax1.set_title('Spectres en amplitude et en phase')
ax1.set_xticks(F)
ax1.legend()
ax1.grid()

ax2 = plt.subplot(212)                                                 # tracé des phases en concordance de f 
ax2.bar(F - largeur/2, phie, largeur, label='phie')                    # spectre pour phie
ax2.bar(F + largeur/2, phis, largeur, label= 'phis pour fc =' + str(fc) + 'Hz') # spectre pour phis
ax2.set_xticks(F)
ax2.set_xlabel('f (Hz)')
ax2.set_ylabel('phase (°)')
ax2.legend(loc='upper right')
ax2.grid()

## Tracés de ue(t) et us(t) avec la même valeur de f0

t = np.linspace(0, 4/fe, 1001)   # tracé entre t = 0 et t(max) = 4T = 4/fe

def ue(t, F) :                   # calcule ue pour chaque instant t
    valeur_ue = 0
    for i in range(len(F)) :    # somme des composantes d'entrée à t
        valeur_ue = valeur_ue \
                    + Ue[i]*np.cos(2*np.pi*F[i]*t + phie[i]/180*np.pi) #rad
    return valeur_ue

def us(t, f0, Us, phis) :  # calcule us pour chaque instant t (Us, phi calculés avant)
    valeur_us = 0
    for i in range(len(F)) :    # somme des composantes de sortie à t
        valeur_us = valeur_us \
                    + Us[i]*np.cos(2*np.pi*F[i]*t + (phie[i]+phis[i])/180*np.pi)  #rad
    return valeur_us

plt.figure(3)
plt.plot(t, ue(t, F), label = 'ue')  # t est une liste, ue(t) aussi
plt.plot(t, us(t, fc, Us, phis), label = 'us')
plt.xlabel('t (en s)'), plt.ylabel('en V')
plt.title('Tracés de ue(t) et us(t) pour fc =' +str(fc) + 'Hz et fe =' +str(fe) + 'Hz')
plt.grid()
plt.legend()
plt.show()